package nemosofts.voxradio.activity;

import android.app.IntentService;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ServiceInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.media.AudioManager;
import android.media.session.MediaSession;
import android.os.Build;
import android.os.CountDownTimer;
import android.os.PowerManager;
import android.support.v4.media.session.MediaSessionCompat;
import android.telephony.TelephonyManager;
import android.util.Log;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;
import androidx.media3.common.AudioAttributes;
import androidx.media3.common.C;
import androidx.media3.common.MediaItem;
import androidx.media3.common.MediaMetadata;
import androidx.media3.common.PlaybackException;
import androidx.media3.common.Player;
import androidx.media3.common.util.UnstableApi;
import androidx.media3.common.util.Util;
import androidx.media3.datasource.DataSource;
import androidx.media3.datasource.DefaultHttpDataSource;
import androidx.media3.exoplayer.ExoPlayer;
import androidx.media3.exoplayer.hls.HlsMediaSource;
import androidx.media3.exoplayer.source.MediaSource;
import androidx.media3.exoplayer.source.ProgressiveMediaSource;
import androidx.media3.exoplayer.upstream.DefaultBandwidthMeter;
import androidx.media3.ui.PlayerNotificationManager;
import androidx.nemosofts.utils.NetworkUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.net.ssl.HttpsURLConnection;

import nemosofts.voxradio.R;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.callback.Method;
import nemosofts.voxradio.item.ItemCat;
import nemosofts.voxradio.item.ItemRadio;
import nemosofts.voxradio.utils.ApplicationUtil;
import nemosofts.voxradio.utils.AsyncTaskExecutor;
import nemosofts.voxradio.utils.AudioRecorder;
import nemosofts.voxradio.utils.GlobalBus;
import nemosofts.voxradio.utils.MessageEvent;
import nemosofts.voxradio.utils.ParserM3UToURL;
import nemosofts.voxradio.utils.helper.DBHelper;
import nemosofts.voxradio.utils.helper.Helper;
import nemosofts.voxradio.utils.helper.SPHelper;
import nemosofts.voxradio.utils.receiver.MediaButtonIntentReceiver;

@UnstableApi
public class PlayerService extends IntentService implements Player.Listener {

    private static final String TAG = "PlayerService";
    public static final String ACTION_TOGGLE = "action.ACTION_TOGGLE";
    public static final String ACTION_PLAY = "action.ACTION_PLAY";
    public static final String ACTION_NEXT = "action.ACTION_NEXT";
    public static final String ACTION_PREVIOUS = "action.ACTION_PREVIOUS";
    public static final String ACTION_STOP = "action.ACTION_STOP";
    public static final String ACTION_SEEKTO = "action.ACTION_SEEKTO";

    public static final String ACTION_START_TIMER = "action.ACTION_START_TIMER";
    public static final String ACTION_STOP_TIMER = "action.ACTION_STOP_TIMER";

    private static ExoPlayer exoPlayer = null;
    MediaSessionCompat mMediaSession;
    DefaultBandwidthMeter.Builder bandwidthMeter;
    DataSource.Factory dataSourceFactory;
    static PlayerService playerService;
    Helper helper;
    SPHelper spHelper;
    DBHelper dbHelper;
    Boolean isNewSong = false;
    Bitmap bitmap;
    ComponentName componentName;
    AudioManager mAudioManager;
    PowerManager.WakeLock mWakeLock;
    PlayerNotificationManager notificationManager;
    NotificationReceiver notificationReceiver;
    PlayerNotificationManager.BitmapCallback callbackBitmap;
    Boolean buffering = false;

    public PlayerService() {
        super(null);
    }

    public static PlayerService getInstance() {
        if (playerService == null) {
            playerService = new PlayerService();
        }
        return playerService;
    }

    public static ExoPlayer getExoPlayer() {
        return exoPlayer;
    }

    @NonNull
    public static Boolean getIsPlaying() {
        return exoPlayer != null && exoPlayer.getPlayWhenReady();
    }

    public long getDuration() {
        if (exoPlayer == null) {
            return 0;
        }
        return exoPlayer.getDuration();
    }

    public int getAudioSessionID() {
        if (exoPlayer == null) {
            return 0;
        }
        return exoPlayer.getAudioSessionId();
    }

    private CountDownTimer countDownTimer;
    private void startTimer(long countdownDuration) {
        long timeLeft = countdownDuration - System.currentTimeMillis();
        countDownTimer = new CountDownTimer(timeLeft, 1000) {
            @Override
            public void onTick(long millisUntilFinished) {
                // this method is empty
            }

            @Override
            public void onFinish() {
                try {
                    Intent intent = new Intent(getApplicationContext(), PlayerService.class);
                    intent.setAction(PlayerService.ACTION_STOP);
                    startService(intent);
                } catch (Exception e) {
                    Log.e(TAG, "Error onFinish Timer");
                }
            }
        };
        countDownTimer.start();
    }

    private void stopTimer() {
        if (countDownTimer != null) {
            countDownTimer.cancel();
            countDownTimer = null;
        }
    }

    @Override
    protected void onHandleIntent(@Nullable Intent intent) {
        // this method is empty
    }

    @Override
    public void onCreate() {

        helper = new Helper(getApplicationContext());
        dbHelper = new DBHelper(getApplicationContext());
        spHelper = new SPHelper(getApplicationContext());

        mAudioManager = (AudioManager) getSystemService(Context.AUDIO_SERVICE);
        mAudioManager.requestAudioFocus(onAudioFocusChangeListener, AudioManager.STREAM_MUSIC, AudioManager.AUDIOFOCUS_GAIN);

        componentName = new ComponentName(getPackageName(), MediaButtonIntentReceiver.class.getName());
        mAudioManager.registerMediaButtonEventReceiver(componentName);

        try {
            registerReceiver(onCallIncome, new IntentFilter("android.intent.action.PHONE_STATE"));
            registerReceiver(onHeadPhoneDetect, new IntentFilter(AudioManager.ACTION_AUDIO_BECOMING_NOISY));
        } catch (Exception e) {
            Log.e(TAG, "Error registering receivers", e);
        }

        bandwidthMeter = new DefaultBandwidthMeter.Builder(this);

        exoPlayer = new ExoPlayer.Builder(getApplicationContext()).build();
        exoPlayer.addListener(this);

        PowerManager powerManager = (PowerManager) getSystemService(Context.POWER_SERVICE);
        mWakeLock = powerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, getClass().getName());
        mWakeLock.setReferenceCounted(false);

        AudioAttributes audioAttributes = new AudioAttributes.Builder()
                .setUsage(C.USAGE_MEDIA)
                .setContentType(C.AUDIO_CONTENT_TYPE_MUSIC)
                .build();
        exoPlayer.setAudioAttributes(audioAttributes, true);

        mMediaSession = new MediaSessionCompat(this, getResources().getString(R.string.app_name));
        mMediaSession.setActive(true);

        notificationReceiver = new NotificationReceiver();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            registerReceiver(notificationReceiver, addIntentFilter(), RECEIVER_EXPORTED);
        } else {
            registerReceiver(notificationReceiver, addIntentFilter());
        }
        super.onCreate();
    }

    @Override
    public int onStartCommand(@Nullable Intent intent, int flags, int startId) {
        if (intent == null) {
            return START_STICKY;
        }
        try {
            String action = intent.getAction();
            if (action == null) {
                return START_STICKY;
            }
            switch (action) {
                case ACTION_START_TIMER:
                    startTimer(spHelper.getSleepTime());
                    break;
                case ACTION_STOP_TIMER:
                    stopTimer();
                    break;
                case ACTION_PLAY :
                    startNewRadio();
                    break;
                case ACTION_TOGGLE :
                    togglePlay();
                    break;
                case ACTION_SEEKTO :
                    long seekPosition = intent.getExtras().getLong("seekto");
                    seekTo(seekPosition);
                    break;
                case ACTION_STOP :
                    stop(intent);
                    break;
                case ACTION_PREVIOUS :
                    handlePrevious();
                    break;
                case ACTION_NEXT :
                    handleNext();
                    break;
                default :
                    break;
            }
        } catch (Exception e) {
            return START_STICKY;
        }
        return START_STICKY;
    }

    private void handlePrevious() {
        if (NetworkUtils.isConnected(this)) {
            navigateToRadio(false);
            exoPlayer.seekToDefaultPosition(Callback.getPlayPos());
        } else {
            Toast.makeText(getApplicationContext(), getResources().getString(R.string.err_internet_not_connected), Toast.LENGTH_SHORT).show();
        }
    }

    private void handleNext() {
        if (NetworkUtils.isConnected(this)) {
            navigateToRadio(true);
            exoPlayer.seekToDefaultPosition(Callback.getPlayPos());
        } else {
            Toast.makeText(getApplicationContext(), getResources().getString(R.string.err_internet_not_connected), Toast.LENGTH_SHORT).show();
        }
    }

    private void createNoti() {
        int notificationId = 111111;

        notificationManager = new PlayerNotificationManager.Builder(this, notificationId, getResources().getString(R.string.app_name))
                .setNotificationListener(notificationListener)
                .setMediaDescriptionAdapter(descriptionAdapter)
                .setChannelImportance(NotificationManager.IMPORTANCE_HIGH)
                .setSmallIconResourceId(R.drawable.ic_audio_track)
                .setChannelDescriptionResourceId(R.string.app_name)
                .setNextActionIconResourceId(R.drawable.ic_skip_next)
                .setPreviousActionIconResourceId(R.drawable.ic_skip_previous)
                .setPlayActionIconResourceId(R.drawable.ic_play)
                .setPauseActionIconResourceId(R.drawable.ic_pause)
                .setChannelNameResourceId(R.string.app_name)
                .build();

        notificationManager.setPlayer(exoPlayer);
        notificationManager.setPriority(NotificationCompat.PRIORITY_MAX);

        notificationManager.setUsePlayPauseActions(true);
        notificationManager.setUseFastForwardAction(false);
        notificationManager.setUseRewindAction(false);
        notificationManager.setUseNextAction(true);
        notificationManager.setUsePreviousAction(true);
        notificationManager.setUseStopAction(true);
        notificationManager.setUseNextActionInCompactView(true);
        notificationManager.setUsePreviousActionInCompactView(true);

        MediaSessionCompat.Token compatToken = mMediaSession.getSessionToken();
        MediaSession.Token frameworkToken = (MediaSession.Token) compatToken.getToken();
        notificationManager.setMediaSessionToken(frameworkToken);

        updateNotiImage();
    }

    PlayerNotificationManager.NotificationListener notificationListener = new PlayerNotificationManager.NotificationListener() {
        @Override
        public void onNotificationPosted(int notificationId, @NonNull Notification notification, boolean ongoing) {
            if (ongoing) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    startForeground(notificationId, notification, ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PLAYBACK);
                } else {
                    startForeground(notificationId, notification);
                }
            }
        }
    };

    PlayerNotificationManager.MediaDescriptionAdapter descriptionAdapter = new PlayerNotificationManager.MediaDescriptionAdapter() {
        @NonNull
        @Override
        public CharSequence getCurrentContentTitle(@NonNull Player player) {
            int playPos = player.getCurrentWindowIndex();
            if (playPos != Callback.getPlayPos()){
                changePlayerPlayPos(playPos);
            }
            return Callback.getArrayListPlay().get(Callback.getPlayPos()).getRadioTitle();
        }

        @Nullable
        @Override
        public PendingIntent createCurrentContentIntent(@NonNull Player player) {
            Intent notificationIntent = new Intent(PlayerService.this, LauncherActivity.class);
            notificationIntent.setAction(Intent.ACTION_MAIN);
            notificationIntent.addCategory(Intent.CATEGORY_LAUNCHER);
            notificationIntent.putExtra("isnoti", true);
            return PendingIntent.getActivity(PlayerService.this, 0, notificationIntent, PendingIntent.FLAG_IMMUTABLE);
        }

        @Nullable
        @Override
        public CharSequence getCurrentContentText(@NonNull Player player) {
            if (Callback.getArrayListPlay() == null || Callback.getPlayPos() < 0) {
                return "";
            }
            return Callback.getArrayListPlay().get(Callback.getPlayPos()).getCategoryName();
        }

        @Nullable
        @Override
        public Bitmap getCurrentLargeIcon(@NonNull Player player, @NonNull PlayerNotificationManager.BitmapCallback callback) {
            callbackBitmap = callback;
            return bitmap;
        }
    };

    @Override
    public void onPlaybackStateChanged(int playbackState) {
        Player.Listener.super.onPlaybackStateChanged(playbackState);
        if (playbackState == Player.STATE_READY) {
            exoPlayer.play();
            if (Boolean.TRUE.equals(isNewSong)) {
                isNewSong = false;
                Callback.setIsPlayed(true);
                setBuffer(false);
                GlobalBus.getBus().postSticky(Callback.getArrayListPlay().get(Callback.getPlayPos()));
                if (notificationManager == null) {
                    createNoti();
                } else {
                    updateNotiImage();
                }
            }
            if (Callback.getIsPlayed() && buffering){
                buffering = false;
                GlobalBus.getBus().postSticky(new MessageEvent(false, "buffering"));
            }
        } else if (playbackState == Player.STATE_BUFFERING) {
            if (Boolean.TRUE.equals(Callback.getIsPlayed())){
                buffering = true;
                GlobalBus.getBus().postSticky(new MessageEvent(true, "buffering"));
            }
        } else if (playbackState == Player.STATE_ENDED) {
            navigateToRadio(true);
        }
    }

    @Override
    public void onIsPlayingChanged(boolean isPlaying) {
        Player.Listener.super.onIsPlayingChanged(isPlaying);
        changePlayPause(isPlaying);
        if (isPlaying) {
            if (!mWakeLock.isHeld()) {
                mWakeLock.acquire(60000);
            }
            return;
        }
        if (mWakeLock.isHeld()) {
            mWakeLock.release();
        }
        isRecord();
    }

    @Override
    public void onPlayerError(@NonNull PlaybackException error) {
        Player.Listener.super.onPlayerError(error);
        Toast.makeText(getApplicationContext(), "Failed : " + error.getErrorCodeName(), Toast.LENGTH_SHORT).show();
        exoPlayer.setPlayWhenReady(false);
        setBuffer(false);
        changePlayPause(false);
        isRecord();
    }

    @Override
    public void onMediaMetadataChanged(@NonNull MediaMetadata mediaMetadata) {
        Player.Listener.super.onMediaMetadataChanged(mediaMetadata);
    }

    @Override
    public void onPositionDiscontinuity(@NonNull Player.PositionInfo oldPosition, @NonNull Player.PositionInfo newPosition, int reason) {
        Player.Listener.super.onPositionDiscontinuity(oldPosition, newPosition, reason);
    }

    private void startNewRadio() {
        isNewSong = true;
        setBuffer(true);
        isRecord();
        try {

            dataSourceFactory = new DefaultHttpDataSource.Factory()
                    .setAllowCrossProtocolRedirects(true)
                    .setKeepPostFor302Redirects(true)
                    .setUserAgent(Util.getUserAgent(getApplicationContext(), "VoxRadioApp"));

            if (Callback.getIsNewAdded() || exoPlayer.getMediaItemCount() == 0
                    || Callback.getPlayPos() >= exoPlayer.getMediaItemCount()) {
                loadMultipleMedia();
            }
            exoPlayer.seekToDefaultPosition(Callback.getPlayPos());
            exoPlayer.prepare();
            exoPlayer.setPlayWhenReady(true);
            addRecent();
        } catch (Exception e) {
            Log.e(TAG, "Error startNewSong", e);
        }
    }

    private void addRecent() {
        try {
            if (Callback.getIsRadio() && !Callback.getArrayListPlay().isEmpty()) {
                if (Boolean.FALSE.equals(Callback.getArrayListPlay().get(Callback.getPlayPos()).getIsPremium())){
                    dbHelper.addToRecent(Callback.getArrayListPlay().get(Callback.getPlayPos()).getId());
                } else {
                    if (spHelper.getIsSubscribed()){
                        dbHelper.addToRecent(Callback.getArrayListPlay().get(Callback.getPlayPos()).getId());
                    }
                }
            }
        } catch (Exception e) {
            Log.e(TAG, "Error addToRecent", e);
        }
    }

    private void loadMultipleMedia() {
        List<MediaSource> mediaItemsSource = new ArrayList<>();
        String description = "Media description for item ";
        for (ItemRadio playerItem : Callback.getArrayListPlay()) {
            MediaSource sampleSource;
            MediaMetadata metadata = new MediaMetadata.Builder()
                    .setTitle(playerItem.getRadioTitle())
                    .setSubtitle(playerItem.getCategoryName())
                    .setDescription(description + playerItem.getRadioTitle())
                    .build();
            String url = playerItem.getRadioUrl();
            if (url.contains(".m3u8")) {
                MediaItem mediaItem = new MediaItem.Builder().setUri(playerItem.getRadioUrl()).setMediaMetadata(metadata).build();
                sampleSource = new HlsMediaSource.Factory(dataSourceFactory).createMediaSource(mediaItem);
            } else if (url.contains(".m3u") || url.contains("yp.shoutcast.com/sbin/tunein-station.m3u?id=")) {
                url = ParserM3UToURL.parse(url, "m3u");
                MediaItem mediaItem = new MediaItem.Builder().setUri(url).setMediaMetadata(metadata).build();
                sampleSource = new ProgressiveMediaSource.Factory(dataSourceFactory).createMediaSource(mediaItem);
            } else if (url.contains(".pls") || url.contains("listen.pls?sid=") || url.contains("yp.shoutcast.com/sbin/tunein-station.pls?id=")) {
                url = ParserM3UToURL.parse(url, "pls");
                MediaItem mediaItem = new MediaItem.Builder().setUri(url).setMediaMetadata(metadata).build();
                sampleSource = new ProgressiveMediaSource.Factory(dataSourceFactory).createMediaSource(mediaItem);
            } else {
                MediaItem mediaItem = new MediaItem.Builder().setUri(url).setMediaMetadata(metadata).build();
                sampleSource = new ProgressiveMediaSource.Factory(dataSourceFactory).createMediaSource(mediaItem);
            }
            mediaItemsSource.add(sampleSource);
        }

        exoPlayer.setMediaSources(mediaItemsSource);
    }

    private void togglePlay() {
        exoPlayer.setPlayWhenReady(!exoPlayer.getPlayWhenReady());
        changePlayPause(exoPlayer.getPlayWhenReady());
        isRecord();
    }

    private void navigateToRadio(boolean isNext) {
        isNewSong = true;
        setBuffer(true);
        int currentPosition = Callback.getPlayPos();
        int playlistSize = Callback.getArrayListPlay().size();
        if (playlistSize > 0) {
            if (isNext) {
                Callback.setPlayPos((currentPosition + 1) % playlistSize); // Wrap around
            } else {
                Callback.setPlayPos((currentPosition > 0) ? currentPosition - 1 : playlistSize - 1);
            }
            handlePlayback();
        } else {
            Log.e(TAG, "Playlist is empty. Cannot navigate to the next radio.");
        }
    }

    private void handlePlayback() {
        if (spHelper.getIsSubscribed()) {
            playOrStartNewRadio();
        } else {
            if (Boolean.TRUE.equals(Callback.getArrayListPlay().get(Callback.getPlayPos()).getIsPremium())) {
                if (spHelper.getRewardCredit() > 0) {
                    spHelper.useRewardCredit(1);
                    playOrStartNewRadio();
                } else {
                    navigateToRadio(true);
                }
            } else {
                playOrStartNewRadio();
            }
        }
    }

    private void playOrStartNewRadio() {
        if (exoPlayer.getMediaItemCount() != 0) {
            exoPlayer.setPlayWhenReady(true);
            return;
        }
        startNewRadio();
    }

    private void seekTo(long seek) {
        exoPlayer.seekTo((int) seek);
    }

    private void changePlayPause(Boolean flag) {
        try {
            changeEqualizer();
            GlobalBus.getBus().postSticky(new MessageEvent(flag, "playicon"));
        } catch (Exception e) {
            Log.e(TAG, "Error changePlayPause", e);
        }
    }

    private void updateNotiImage() {
        new AsyncTaskExecutor<String, String, String>() {
            @Override
            protected void onPostExecute(String s) {
                // this method is empty
            }

            @Override
            protected String doInBackground(String strings) {
                try {
                    if(Boolean.TRUE.equals(Callback.getIsRadio())) {
                        ApplicationUtil.responsePost(Callback.API_URL, helper.getAPIRequest(Method.METHOD_SINGLE_RADIO,0,
                                Callback.getArrayListPlay().get(Callback.getPlayPos()).getId(),"",
                                "","","","","","", "",
                                "","", "", null)
                        );
                    }
                    getBitmapFromURL(Callback.getArrayListPlay().get(Callback.getPlayPos()).getImage());
                    return "1";
                } catch (Exception e) {
                    return "0";
                }
            }
        }.execute();
    }

    private void setBuffer(Boolean isBuffer) {
        if (Boolean.FALSE.equals(isBuffer)) {
            changeEqualizer();
        }
        GlobalBus.getBus().postSticky(new MessageEvent(isBuffer, "buffer"));
    }

    private void changeEqualizer() {
        GlobalBus.getBus().postSticky(new ItemCat("", "", null));
    }

    public void changePlayerPlayPos(int playPos) {
        addRecent();
        try {
            Callback.setPlayPos(playPos);
            GlobalBus.getBus().postSticky(Callback.getArrayListPlay().get(Callback.getPlayPos()));
            if (notificationManager != null) {
                updateNotiImage();
            }
            changeEqualizer();
        } catch (Exception e) {
            Log.e(TAG, "Error changePlayerPlayPos", e);
        }
    }

    private void getBitmapFromURL(String src) {
        try {
            URL url = new URL(src);
            InputStream input;
            if (src.contains("https://")) {
                HttpsURLConnection connection = (HttpsURLConnection) url.openConnection();
                connection.setDoInput(true);
                connection.connect();
                input = connection.getInputStream();
            } else {
                HttpURLConnection connection = (HttpURLConnection) url.openConnection();
                connection.setDoInput(true);
                connection.connect();
                input = connection.getInputStream();
            }
            bitmap = BitmapFactory.decodeStream(input);
            if (callbackBitmap != null) {
                callbackBitmap.onBitmap(bitmap);
            }
        } catch (IOException e) {
            Log.e(TAG, "Error getBitmapFromURL", e);
        }
    }

    BroadcastReceiver onCallIncome = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, @NonNull Intent intent) {
            try {
                String a = intent.getStringExtra(TelephonyManager.EXTRA_STATE);
                if (a != null && exoPlayer.getPlayWhenReady()
                        && (a.equals(TelephonyManager.EXTRA_STATE_OFFHOOK)
                        || a.equals(TelephonyManager.EXTRA_STATE_RINGING))) {
                    exoPlayer.setPlayWhenReady(false);
                }
            } catch (Exception e) {
                Log.e(TAG, "Error onCallIncome", e);
            }
        }
    };

    BroadcastReceiver onHeadPhoneDetect = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            try {
                if (exoPlayer.getPlayWhenReady()) {
                    togglePlay();
                }
            } catch (Exception e) {
                Log.e(TAG, "Error onHeadPhoneDetect", e);
            }
        }
    };

    AudioManager.OnAudioFocusChangeListener onAudioFocusChangeListener = focusChange -> {
        if (focusChange == AudioManager.AUDIOFOCUS_LOSS || focusChange == AudioManager.AUDIOFOCUS_LOSS_TRANSIENT){
            try {
                if (exoPlayer.getPlayWhenReady()) {
                    togglePlay();
                }
            } catch (Exception e) {
                Log.e(TAG, "Error onAudioFocusChangeListener", e);
            }
        }
    };

    private class NotificationReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, @NonNull Intent intent) {
            String action = intent.getAction();
            if (action == null) {
                return;
            }
            switch (Objects.requireNonNull(intent.getAction())) {
                case PlayerNotificationManager.ACTION_PREVIOUS :
                    navigateToRadio(false);
                    break;
                case PlayerNotificationManager.ACTION_NEXT :
                    navigateToRadio(true);
                    break;
                case PlayerNotificationManager.ACTION_STOP :
                    stop(intent);
                    break;
                default:
                    break;
            }
        }
    }

    @NonNull
    private IntentFilter addIntentFilter() {
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(PlayerNotificationManager.ACTION_PREVIOUS);
        intentFilter.addAction(PlayerNotificationManager.ACTION_NEXT);
        intentFilter.addAction(PlayerNotificationManager.ACTION_NEXT);
        intentFilter.addAction(PlayerNotificationManager.ACTION_STOP);
        return intentFilter;
    }

    private void stop(Intent intent) {
        try {
            if (Boolean.TRUE.equals(new SPHelper(this).getIsSleepTimeOn())) {
                spHelper.setSleepTime(false, 0, 0);
            }
            Callback.setIsPlayed(false);
            exoPlayer.setPlayWhenReady(false);
            changePlayPause(false);
            exoPlayer.stop();
            exoPlayer.release();
            exoPlayer = null;
            isRecord();
        } catch (Exception e) {
            Log.e(TAG, "Error stop", e);
        }

        try {
            mAudioManager.abandonAudioFocus(onAudioFocusChangeListener);
            mAudioManager.unregisterMediaButtonEventReceiver(componentName);
            unregisterReceiver(onCallIncome);
            unregisterReceiver(onHeadPhoneDetect);
        } catch (Exception e) {
            Log.e(TAG, "Error Receiver AudioManager", e);
        }

        try {
            stopService(intent);
            stopForeground(true);
            stopSelf();
        } catch (Exception e) {
            Log.e(TAG, "Error stop", e);
        }
    }

    @Override
    public void onDestroy() {
        try {
            if (notificationManager != null) {
                notificationManager.setPlayer(null);
                notificationManager = null;
            }

            try {
                isRecord();
                Callback.setIsPlayed(false);
                try {
                    mAudioManager.abandonAudioFocus(onAudioFocusChangeListener);
                    mAudioManager.unregisterMediaButtonEventReceiver(componentName);
                    unregisterReceiver(onCallIncome);
                    unregisterReceiver(onHeadPhoneDetect);
                } catch (Exception e) {
                    Log.e(TAG, "Error onDestroy", e);
                }
                stopForeground(true);
                stopSelf();
            } catch (Exception e) {
                Log.e(TAG, "Error onDestroy", e);
            }

            unregisterReceiver(notificationReceiver);
            if (mWakeLock.isHeld()) {
                mWakeLock.release();
            }
            dbHelper.close();
        } catch (Exception e) {
            Log.e(TAG, "Error onDestroy", e);
        }
        super.onDestroy();
    }

    private void isRecord(){
        if (AudioRecorder.isRecording()){
            AudioRecorder.stopRecording();
        }
    }

    @Override
    public void onTaskRemoved(Intent rootIntent) {
        super.onTaskRemoved(rootIntent);
        try {
            if (!exoPlayer.getPlayWhenReady())
                stopForeground(true);
        } catch (Exception e) {
            Log.e(TAG, "Error onTaskRemoved", e);
        }
    }
}